'''GUI elements that are not a part of xi proper, but are widely useful anyway.
'''

import ika
from xi import gui

def init(font, wnd=None, csr=None):
    '''Initializes defaults for the GUI system.  This MUST be called
       before creating any instances.
    '''
    global default_font, default_window, default_cursor
    default_font = font
    default_window = wnd
    default_cursor = csr


class TextFrame(gui.Frame):
    '''A frame with text in it.  This is a simple convenience class,
       combining the Frame and StaticText controls into a single
       convenient object.
    '''
    def __init__(self, x=0, y=0, width=0, height=0, *args, **kwargs):
        super(TextFrame, self).__init__(x, y, width, height, *args, **kwargs)
        # way cool.  since keyword arguments are passed on, the font will be
        # set properly.  additionally, text will be added just like StaticText.
        # Consistency totally rules.
        self.__text = gui.StaticText(0, 0, width, height, *args, **kwargs)
        self.addChild(self.__text)
        self.autoSize()

    def getText(self):
        return self.__text.text

    def setText(self, value):
        self.__text.setText(value)

    def getFont(self):
        return self.__text.font

    def setFont(self, font):
        self.__text.font = font

    text = property(
        fget=getText,
        fset=setText,
        doc='Gets or sets the text contained by the control.'
    )

    font = property(
        fget=getFont,
        fset=setFont,
        doc='Gets or sets the font used for the text contained by the control.'
    )

    def addText(self, *args):
        '''Appends text to what is already contained.'''
        self.text.addText(*args)

    def autoSize(self):
        '''Autosizes the frame such that it is just large enough to
           contain its text.
        '''
        self.__text.autoSize()
        self.size = self.__text.size


class ScrollableTextFrame(gui.ScrollableTextLabel):
    def __init__(self, *args, **kwargs):
        super(ScrollableTextFrame, self).__init__(self, *args, **kwargs)

        self.window = kwargs.get('window', gui.default_window)
        self.border = self.window.border * 2

    def draw(self, x=0, y=0):
        self.window.draw(x, y, self.width, self.height)
        super(ScrollableTextFrame, self).draw(x, y)

