
import ika

from xi import gui
from xi import cursor
from xi import layout
from xi.menu import Menu, Cancel
from xi.window import ImageWindow
from xi.transition import Transition, WindowMover

import engine
import effects
import config
import controls
import sound
import saveloadmenu

from gui import ScrollableTextFrame
from gameover import EndGameException

class Window(object):
    '''Specialized window.  Ghastly and hacked up to accomodate the awesome
    looking window frame thingie that corey drew.
    '''

    def __init__(self, nameTemplate):
        super(Window, self).__init__()

        self.iTopleft, self.iTopright, self.iBottomleft, self.iBottomright = [
            ika.Image(nameTemplate % i) for i in
            ('top_left', 'top_right', 'bottom_left', 'bottom_right')
        ]

        self.iLeft, self.iRight, self.iTop, self.iBottom = [
            ika.Image(nameTemplate % i) for i in
            ('left', 'right', 'top', 'bottom')
        ]

        self.Blit = ika.Video.ScaleBlit
        self.border = max(self.iLeft.width, self.iTop.height)

    def draw(self, x, y, w, h):
        b = self.left / 2
        x2 = x + w + b
        y2 = y + h + b
        x -= b
        y -= b

        ika.Video.Blit(self.iTopright, x2, y - self.iTopright.height)
        ika.Video.Blit(self.iBottomleft, x - self.iBottomleft.width, y2)
        self.Blit(self.iLeft, x - self.iLeft.width, y, self.iLeft.width, y2 - y)
        self.Blit(self.iRight, x2, y, self.iRight.width, y2 - y)
        self.Blit(self.iTop, x, y - self.iTop.height, x2 - x, self.iTop.height)
        self.Blit(self.iBottom, x, y2, x2 - x, self.iBottom.height)
        ika.Video.DrawRect(x, y, x2, y2, ika.RGB(0,0,0), True)

        # EVIL HARD CODED
        ax = 8
        ay = 12

        # These two need special treatment, 'cause they're shiny and huge
        ika.Video.Blit(self.iTopleft,  x - ax, y - ay)
        ika.Video.Blit(self.iBottomright, x2 - self.iBottomright.width + ax, y2 - self.iBottomright.height + ay)

    left   = property(lambda self: self.border)
    right  = property(lambda self: self.border)
    top    = property(lambda self: self.border)
    bottom = property(lambda self: self.border)

class SubScreenWindow(gui.Frame):

    def __init__(self, *args, **kw):
        super(SubScreenWindow, self).__init__(*args, **kw)
        self.layout = self.createLayout()
        self.addChild(self.layout)

    def createLayout(self):
        return layout.VerticalBoxLayout()

    def refresh(self):
        stats = engine.player.stats
        self.layout.setChildren(self.createContents())
        self.layout.layout()
        self.autoSize()

class StatWindow(SubScreenWindow):

    def createContents(self):
        stats = engine.player.stats
        return (
            gui.StaticText(text='Anastasia'),
            gui.StaticText(text=''),
            gui.StaticText(text='Level:\t\t%03i' % stats.level),
            gui.StaticText(text=''),
            gui.StaticText(text='Experience:'),
            gui.StaticText(text='%06i/%06i' % (stats.exp, stats.next)),
            gui.StaticText(text=''),
            gui.StaticText(text='HP:\t%03i/%03i' % (stats.hp, stats.maxhp)),
            gui.StaticText(text=''),
            gui.StaticText(text='Attack:\t\t%03i' % stats.att),
            gui.StaticText(text='Magic:\t\t%03i' % stats.mag),
            gui.StaticText(text='Defense:\t\t%03i' % stats.pres),
            gui.StaticText(text='Resistance:\t%03i' % stats.mres),
            gui.StaticText(text=''),
            gui.StaticText(text='Seashells:\t%i' % stats.money)
        )


class AttribWindow(SubScreenWindow):

    def __init__(self):
        super(AttribWindow, self).__init__()

    def createContents(self):
        stats = engine.player.stats
        return (
            gui.StaticText(text='Attack:\t\t%03i' % stats.att),
            gui.StaticText(text='Magic:\t\t%03i' % stats.mag),
            gui.StaticText(text='Defense:\t\t%03i' % stats.pres),
            gui.StaticText(text='Resistance:\t%03i' % stats.mres)
        )


class ToolWindow(SubScreenWindow):

    #ITEM_NAMES = ('spear', 'sword', 'glove', 'grapple', 'petrify', 'wind', 'heal')
    #WEAPONS = ('spear', 'sword', 'glove')
    ITEM_NAMES = ('spear', 'sword', 'grapple')
    WEAPONS = ('spear', 'sword')

    def __init__(self):
        super(ToolWindow, self).__init__()

        self.icons = dict(
            (name, ika.Image('gfx/ui/icon_%s.png' % name)) for name in self.ITEM_NAMES
        )

        self.cursor = ika.Image('gfx/ui/hud_box.png')
        self.cursorPos = 0

        self.layout2 = layout.FlexGridLayout(cols=len(self.icons), pad=4)

    def createLayout(self):
        return layout.VerticalBoxLayout(pad=0)

    def createContents(self):
        player = engine.player
        stats = player.stats

        size = 16, 16

        contents = []
        for name in self.ITEM_NAMES:
            icon = self.icons[name]

            if engine.saveData.get(name) is not None:
                contents.append(gui.Picture(image=icon, width=size[0], height=size[1]))
            else:
                contents.append(layout.Spacer(width=size[0], height=size[1]))

        i = 0
        for name in self.ITEM_NAMES:
            s = ''

            if engine.saveData.get(name):
                if name in self.WEAPONS:
                    if player.weapon is getattr(player, name):
                        s = '*'
                else:
                    s = '  DSA123456789'[i]

            i += 1
            contents.append(gui.StaticText(text=s.center(3)))

        self.layout2.setChildren(contents)

        return (
            gui.StaticText(text='Tools'),
            self.layout2,
        )

    def drawCursor(self, active=False):
        pic = self.layout2.children[self.cursorPos]

        WIDTH = 4
        x = pic.x + self.x + self.layout2.x - WIDTH
        y = pic.y + self.y + self.layout2.y - WIDTH

        BLINK_RATE = 50
        blink = (ika.GetTime() % BLINK_RATE) * 2 > BLINK_RATE

        if active and blink:
            ika.Video.Blit(self.cursor, x, y)
        else:
            ika.Video.TintBlit(self.cursor, x, y, ika.RGB(128, 128, 128))

    def update(self):
        if controls.left.pressed:
            self.cursorPos = max(0, self.cursorPos - 1)
            sound.menuMove.Play()

        elif controls.right.pressed:
            self.cursorPos = min(len(self.WEAPONS) - 1, self.cursorPos + 1)
            sound.menuMove.Play()

        elif controls.attack1.pressed:
            # have to use pressed and not position here, because the other menu eats it
            self.select(self.cursorPos)

    def select(self, cursorPos):
        player = engine.player

        name = self.ITEM_NAMES[cursorPos]

        if name not in engine.saveData: return # can't use something you don't have

        sound.menuSelect.Play()

        if name in self.WEAPONS and player.weapon is not getattr(player, name):
            player.weapon =  getattr(player, name)
            self.refresh()
        else:
            return # TODO someday: have items that can be used from the subscreen?


class MenuWindow(Menu):

    def __init__(self):
        super(MenuWindow, self).__init__(textctrl=gui.ScrollableTextLabel())
        self.addText(
            'Resume',
            'Equip',
            'Exit'
        )
        self.autoSize()


class PauseScreen(object):

    def __init__(self):
        assert _initted
        super(PauseScreen, self).__init__()
        self.statWnd = StatWindow()
        self.toolWnd = ToolWindow()
        self.menu = MenuWindow()

        self.menuWindow = gui.Frame()
        self.menuWindow.addChild(self.menu)
        self.menuWindow.autoSize()

    def refresh(self):
        self.statWnd.refresh()
        self.statWnd.dockTop().dockLeft()
        self.toolWnd.refresh()
        self.toolWnd.dockBottom().dockRight()
        self.menuWindow.dockTop().dockRight()

    def show(self):
        TIME = 35
        self.refresh()

        t = Transition()

        t.addChild(self.statWnd,
            startRect=(-self.statWnd.right, self.statWnd.top),
            time=TIME
        )

        t.addChild(self.toolWnd,
            startRect=(ika.Video.xres, self.toolWnd.top),
            time=TIME
        )

        t.addChild(self.menuWindow,
            startRect=(ika.Video.xres, self.menuWindow.top),
            time=TIME
        )

        for i in range(TIME):
            t.update(1)
            o = i * 128 / TIME  # tint intensity for this frame
            engine.raw_draw()
            ika.Video.DrawRect(0, 0, ika.Video.xres, ika.Video.yres,
                               ika.RGB(0, 0, 0, o), True)
            self.draw()
            ika.Video.ShowPage()
            ika.Input.Update()

    def hide(self):
        TIME = 40
        t = Transition()

        t.addChild(
            self.statWnd,
            endRect=(-self.statWnd.right, self.statWnd.top),
            time=TIME
        )

        t.addChild(
            self.toolWnd,
            endRect=(ika.Video.xres, self.toolWnd.top),
            time=TIME
        )

        t.addChild(
            self.menuWindow,
            endRect=(ika.Video.xres, self.menuWindow.top),
            time=TIME
        )

        for i in range(TIME - 1, -1, -1):
            t.update(1)
            o = i * 255 / TIME  # menu opacity for this frame
            engine.raw_draw()
            ika.Video.DrawRect(0, 0, ika.Video.xres, ika.Video.yres,
                               ika.RGB(0, 0, 0, o / 2), True)
            self.draw(o)
            ika.Video.ShowPage()
            ika.Input.Update()

    def draw(self, opacity=255):
        gui.default_window.opacity = opacity
        self.statWnd.draw()
        self.toolWnd.draw()
        self.menuWindow.draw()

        self.toolWnd.drawCursor(self.menu.cursorPos == 1)

    def run(self):
        self.show()
        while True:
            engine.raw_draw()

            ika.Video.DrawRect(
                0, 0, ika.Video.xres, ika.Video.yres,
                ika.RGB(0, 0, 0, 128), True
            )

            self.draw()
            ika.Video.ShowPage()
            ika.Input.Update()

            if self.menu.cursorPos == 1:
                self.toolWnd.update()

            result = self.menu.update()

            if result is None:
                continue

            elif result is Cancel or result == 0:
                self.hide()
                return

            elif result == 1: # change equip: do nothing, this just makes the inventory cursor active
                continue

            elif result == 2: # Exit
                self.hide()
                self.exitGame()
                return

        assert False, 'Execution should never reach this point.'

    def exitGame(self):
        # TODO: shiny fade out
        raise EndGameException

_initted = False

def init():
    global _initted
    _initted = True
    gui.init(
        font=ika.Font(config.FONT),
        window=Window('gfx/ui/win_%s.png'),
        cursor=cursor.ImageCursor('gfx/ui/pointer.png', hotspot=(14, 6))
    )
